// server/index.ts
import express2 from "express";

// server/routes.ts
import { createServer } from "http";

// server/storage.ts
var MemStorage = class {
  users;
  services;
  blogPosts;
  testimonials;
  contactMessages;
  appointments;
  youtubeVideos;
  userCurrentId;
  serviceCurrentId;
  blogPostCurrentId;
  testimonialCurrentId;
  contactMessageCurrentId;
  appointmentCurrentId;
  youtubeVideoCurrentId;
  constructor() {
    this.users = /* @__PURE__ */ new Map();
    this.services = /* @__PURE__ */ new Map();
    this.blogPosts = /* @__PURE__ */ new Map();
    this.testimonials = /* @__PURE__ */ new Map();
    this.contactMessages = /* @__PURE__ */ new Map();
    this.appointments = /* @__PURE__ */ new Map();
    this.youtubeVideos = /* @__PURE__ */ new Map();
    this.userCurrentId = 1;
    this.serviceCurrentId = 1;
    this.blogPostCurrentId = 1;
    this.testimonialCurrentId = 1;
    this.contactMessageCurrentId = 1;
    this.appointmentCurrentId = 1;
    this.youtubeVideoCurrentId = 1;
    this.initializeData();
  }
  // User methods
  async getUser(id) {
    return this.users.get(id);
  }
  async getUserByUsername(username) {
    return Array.from(this.users.values()).find(
      (user) => user.username === username
    );
  }
  async createUser(insertUser) {
    const id = this.userCurrentId++;
    const user = { ...insertUser, id };
    this.users.set(id, user);
    return user;
  }
  // Service methods
  async getServices() {
    return Array.from(this.services.values());
  }
  async getServiceById(id) {
    return this.services.get(id);
  }
  async getServiceBySlug(slug) {
    return Array.from(this.services.values()).find(
      (service) => service.slug === slug
    );
  }
  async createService(insertService) {
    const id = this.serviceCurrentId++;
    const service = { ...insertService, id };
    this.services.set(id, service);
    return service;
  }
  // Blog post methods
  async getBlogPosts() {
    return Array.from(this.blogPosts.values()).sort((a, b) => new Date(b.publishedAt).getTime() - new Date(a.publishedAt).getTime());
  }
  async getBlogPostById(id) {
    return this.blogPosts.get(id);
  }
  async getBlogPostBySlug(slug) {
    return Array.from(this.blogPosts.values()).find(
      (post) => post.slug === slug
    );
  }
  async createBlogPost(insertBlogPost) {
    const id = this.blogPostCurrentId++;
    const post = {
      ...insertBlogPost,
      id,
      publishedAt: /* @__PURE__ */ new Date(),
      published: insertBlogPost.published === void 0 ? true : insertBlogPost.published
    };
    this.blogPosts.set(id, post);
    return post;
  }
  // Testimonial methods
  async getTestimonials() {
    return Array.from(this.testimonials.values());
  }
  async getTestimonialById(id) {
    return this.testimonials.get(id);
  }
  async createTestimonial(insertTestimonial) {
    const id = this.testimonialCurrentId++;
    const testimonial = { ...insertTestimonial, id };
    this.testimonials.set(id, testimonial);
    return testimonial;
  }
  // Contact message methods
  async createContactMessage(insertMessage) {
    const id = this.contactMessageCurrentId++;
    const message = {
      ...insertMessage,
      id,
      createdAt: /* @__PURE__ */ new Date(),
      phone: insertMessage.phone === void 0 ? null : insertMessage.phone
    };
    this.contactMessages.set(id, message);
    return message;
  }
  async getContactMessages() {
    return Array.from(this.contactMessages.values()).sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
  }
  // Appointment methods
  async createAppointment(insertAppointment) {
    const id = this.appointmentCurrentId++;
    const appointment = {
      ...insertAppointment,
      id,
      createdAt: /* @__PURE__ */ new Date(),
      message: insertAppointment.message === void 0 ? null : insertAppointment.message,
      serviceId: insertAppointment.serviceId === void 0 ? null : insertAppointment.serviceId
    };
    this.appointments.set(id, appointment);
    return appointment;
  }
  async getAppointments() {
    return Array.from(this.appointments.values()).sort((a, b) => new Date(b.createdAt).getTime() - new Date(a.createdAt).getTime());
  }
  async getAppointmentById(id) {
    return this.appointments.get(id);
  }
  // YouTube video methods
  async getYoutubeVideos() {
    return Array.from(this.youtubeVideos.values()).sort((a, b) => new Date(b.publishedAt).getTime() - new Date(a.publishedAt).getTime());
  }
  async getYoutubeVideoById(id) {
    return this.youtubeVideos.get(id);
  }
  async getYoutubeVideoByVideoId(videoId) {
    return Array.from(this.youtubeVideos.values()).find(
      (video) => video.videoId === videoId
    );
  }
  async createYoutubeVideo(insertVideo) {
    const id = this.youtubeVideoCurrentId++;
    const video = {
      ...insertVideo,
      id,
      publishedAt: /* @__PURE__ */ new Date(),
      description: insertVideo.description === void 0 ? null : insertVideo.description,
      thumbnailUrl: insertVideo.thumbnailUrl === void 0 ? null : insertVideo.thumbnailUrl,
      category: insertVideo.category === void 0 ? null : insertVideo.category
    };
    this.youtubeVideos.set(id, video);
    return video;
  }
  async getYoutubeVideosByCategory(category) {
    return Array.from(this.youtubeVideos.values()).filter((video) => video.category === category).sort((a, b) => new Date(b.publishedAt).getTime() - new Date(a.publishedAt).getTime());
  }
  // Initialize with sample data
  initializeData() {
    const servicesData = [
      {
        title: "Sports Injuries",
        description: "Specialized treatment for athletes and active individuals to help you recover and return to your sport quickly and safely.",
        icon: "fa-dumbbell",
        slug: "sports-injuries"
      },
      {
        title: "Back & Neck Pain",
        description: "Effective treatment approaches for chronic and acute back and neck pain to improve mobility and reduce discomfort.",
        icon: "fa-spine",
        slug: "back-neck-pain"
      },
      {
        title: "Joint Pain",
        description: "Personalized treatment plans for knee, shoulder, hip and other joint pain, focusing on improved function and pain relief.",
        icon: "fa-bone",
        slug: "joint-pain"
      },
      {
        title: "Post-Surgical Rehabilitation",
        description: "Comprehensive rehabilitation programs for patients recovering from surgery to restore function and strength.",
        icon: "fa-hospital",
        slug: "post-surgical-rehabilitation"
      },
      {
        title: "Gait Analysis",
        description: "Expert analysis of your walking pattern to identify issues and develop corrective strategies for better mobility.",
        icon: "fa-walking",
        slug: "gait-analysis"
      },
      {
        title: "Manual Therapy",
        description: "Hands-on techniques to reduce pain, improve joint mobility, and address muscle tension for better overall function.",
        icon: "fa-hands",
        slug: "manual-therapy"
      }
    ];
    servicesData.forEach((service) => {
      this.createService(service);
    });
    const blogPostsData = [
      {
        title: "5 Effective Exercises for Relieving Back Pain",
        content: `
          <p>Back pain is one of the most common reasons people seek physiotherapy. Whether it's from poor posture, injury, or a chronic condition, the discomfort can significantly impact your quality of life.</p>
          <p>Here are five exercises that can help alleviate back pain:</p>
          <h3>1. Pelvic Tilts</h3>
          <p>This gentle exercise helps strengthen the abdominal muscles and stretch the lower back.</p>
          <ul>
            <li>Lie on your back with knees bent and feet flat on the floor</li>
            <li>Tighten your abdominal muscles and flatten your back against the floor</li>
            <li>Hold for 5 seconds, then release</li>
            <li>Repeat 10 times</li>
          </ul>
          <h3>2. Cat-Cow Stretch</h3>
          <p>This yoga-inspired movement improves flexibility and mobility in the spine.</p>
          <ul>
            <li>Start on your hands and knees in a tabletop position</li>
            <li>Inhale, dropping your belly towards the floor and lifting your gaze (Cow)</li>
            <li>Exhale, arching your spine towards the ceiling and tucking your chin (Cat)</li>
            <li>Repeat 10-15 times, moving with your breath</li>
          </ul>
          <h3>3. Bridge Exercise</h3>
          <p>The bridge strengthens the glutes and lower back while stretching the hip flexors.</p>
          <ul>
            <li>Lie on your back with knees bent and feet flat on the floor</li>
            <li>Keeping your shoulders on the floor, lift your hips towards the ceiling</li>
            <li>Hold for 5 seconds, then lower slowly</li>
            <li>Repeat 10 times</li>
          </ul>
          <h3>4. Bird Dog</h3>
          <p>This exercise improves core stability and strengthens back muscles.</p>
          <ul>
            <li>Start on your hands and knees</li>
            <li>Simultaneously extend your right arm forward and left leg backward</li>
            <li>Hold for 5 seconds, maintaining balance</li>
            <li>Return to starting position and switch sides</li>
            <li>Repeat 10 times on each side</li>
          </ul>
          <h3>5. Child's Pose</h3>
          <p>A gentle stretch that helps relieve tension in the back and promotes relaxation.</p>
          <ul>
            <li>Kneel on the floor with knees hip-width apart</li>
            <li>Sit back on your heels and reach your arms forward</li>
            <li>Rest your forehead on the floor and relax</li>
            <li>Hold for 30 seconds to 1 minute</li>
          </ul>
          <p>Remember to perform these exercises slowly and with control. If you experience increased pain, stop immediately and consult with a physiotherapist.</p>
          <p>For a personalized exercise program tailored to your specific condition, book an appointment with one of our experienced physiotherapists at PhysioForU.</p>
        `,
        excerpt: "Learn simple yet effective exercises that can help alleviate back pain and improve your spinal health.",
        imageUrl: "https://images.unsplash.com/photo-1573495804664-b1c0849525af?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=500&q=80",
        slug: "5-effective-exercises-for-relieving-back-pain",
        published: true
      },
      {
        title: "Physiotherapy Techniques You Can Do at Home",
        content: `
          <p>Physiotherapy is most effective when combined with a consistent home exercise program. While professional sessions are important, the exercises you do at home can significantly impact your recovery speed and outcomes.</p>
          <p>Here are some effective physiotherapy techniques you can safely perform at home:</p>
          <h3>Self-Massage with a Foam Roller</h3>
          <p>Foam rolling helps release muscle tension and improve circulation.</p>
          <ul>
            <li>Place the foam roller under the target muscle group</li>
            <li>Slowly roll back and forth, pausing on tender areas</li>
            <li>Spend 1-2 minutes per muscle group</li>
            <li>Focus on larger muscles like quads, hamstrings, and back</li>
          </ul>
          <h3>Gentle Stretching Routine</h3>
          <p>Regular stretching improves flexibility and reduces muscle stiffness.</p>
          <ul>
            <li>Hold each stretch for 20-30 seconds</li>
            <li>Breathe deeply and relax into the stretch</li>
            <li>Avoid bouncing or forcing the stretch</li>
            <li>Include stretches for major muscle groups</li>
          </ul>
          <h3>Balance Exercises</h3>
          <p>These exercises improve stability and proprioception, which is especially important after injuries.</p>
          <ul>
            <li>Stand on one foot for 30 seconds</li>
            <li>Progress to closing your eyes while balancing</li>
            <li>Use a wall or chair for support if needed</li>
            <li>Practice daily to see improvements</li>
          </ul>
          <h3>Resistance Band Exercises</h3>
          <p>Resistance bands are affordable and versatile tools for strengthening.</p>
          <ul>
            <li>Use for strengthening shoulders, hips, and knees</li>
            <li>Start with 10-15 repetitions per exercise</li>
            <li>Focus on proper form rather than quantity</li>
            <li>Increase resistance as you get stronger</li>
          </ul>
          <h3>Posture Correction</h3>
          <p>Good posture is crucial for preventing and managing many conditions.</p>
          <ul>
            <li>Set reminders to check your posture throughout the day</li>
            <li>Practice sitting and standing with proper alignment</li>
            <li>Strengthen core muscles to support good posture</li>
            <li>Consider ergonomic adjustments to your workspace</li>
          </ul>
          <p>Important: Always follow the specific instructions from your physiotherapist, as they will tailor exercises to your condition and stage of recovery. If you experience increased pain or discomfort with any exercise, stop and consult your healthcare provider.</p>
          <p>For a personalized home exercise program, book an appointment with PhysioForU today.</p>
        `,
        excerpt: "Discover easy-to-follow physiotherapy methods that you can incorporate into your daily routine at home.",
        imageUrl: "https://images.unsplash.com/photo-1517836357463-d25dfeac3438?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=500&q=80",
        slug: "physiotherapy-techniques-you-can-do-at-home",
        published: true
      },
      {
        title: "Understanding Post-Surgery Rehabilitation",
        content: `
          <p>Surgery is often just the first step in your recovery journey. The rehabilitation process that follows is equally important for achieving optimal outcomes and returning to your normal activities.</p>
          <h3>Why is Post-Surgery Rehabilitation Important?</h3>
          <p>Rehabilitation after surgery serves multiple critical purposes:</p>
          <ul>
            <li>Minimizes pain and swelling</li>
            <li>Prevents complications like blood clots and muscle atrophy</li>
            <li>Restores normal range of motion and flexibility</li>
            <li>Rebuilds strength and endurance</li>
            <li>Improves functional abilities for daily activities</li>
            <li>Facilitates faster and more complete recovery</li>
          </ul>
          <h3>The Phases of Post-Surgery Rehabilitation</h3>
          <p><strong>Phase 1: Acute Recovery (1-2 weeks post-surgery)</strong></p>
          <p>This initial phase focuses on:</p>
          <ul>
            <li>Pain management</li>
            <li>Controlling swelling</li>
            <li>Protecting the surgical site</li>
            <li>Gentle movement as approved by your surgeon</li>
            <li>Basic exercises to prevent complications</li>
          </ul>
          <p><strong>Phase 2: Motion and Basic Strengthening (2-6 weeks post-surgery)</strong></p>
          <p>As healing progresses, rehabilitation shifts to:</p>
          <ul>
            <li>Increasing range of motion</li>
            <li>Beginning light resistance exercises</li>
            <li>Improving balance and coordination</li>
            <li>Reducing dependence on assistive devices</li>
            <li>More focus on functional movements</li>
          </ul>
          <p><strong>Phase 3: Advanced Strengthening and Functional Training (6-12 weeks post-surgery)</strong></p>
          <p>This phase typically includes:</p>
          <ul>
            <li>Progressive resistance training</li>
            <li>More challenging balance exercises</li>
            <li>Activity-specific training</li>
            <li>Cardiovascular conditioning</li>
            <li>Preparation for return to work or sports</li>
          </ul>
          <p><strong>Phase 4: Return to Normal Activities (3+ months post-surgery)</strong></p>
          <p>The final phase focuses on:</p>
          <ul>
            <li>Sport-specific or job-specific training</li>
            <li>Advanced strength and conditioning</li>
            <li>Injury prevention strategies</li>
            <li>Maintenance programs</li>
          </ul>
          <h3>What to Expect During Post-Surgery Physiotherapy</h3>
          <p>Your physiotherapy program will be tailored to your specific surgery, condition, and goals, but generally includes:</p>
          <ul>
            <li>Regular sessions with a physiotherapist</li>
            <li>A customized home exercise program</li>
            <li>Manual therapy techniques</li>
            <li>Modalities for pain and inflammation (ice, heat, electrotherapy)</li>
            <li>Progression assessments</li>
            <li>Education on activity modifications</li>
          </ul>
          <p>At PhysioForU, our experienced physiotherapists work closely with your surgical team to ensure a coordinated approach to your recovery. We understand that every patient and procedure is unique, and we tailor our rehabilitation programs accordingly.</p>
          <p>If you're preparing for surgery or are in the post-surgical recovery phase, contact us to discuss how our rehabilitation services can support your journey back to health.</p>
        `,
        excerpt: "A comprehensive guide to what you can expect during rehabilitation after surgery and how to maximize recovery.",
        imageUrl: "https://images.unsplash.com/photo-1599058917765-a780eda07a3e?ixlib=rb-4.0.3&auto=format&fit=crop&w=800&h=500&q=80",
        slug: "understanding-post-surgery-rehabilitation",
        published: true
      }
    ];
    blogPostsData.forEach((post) => {
      this.createBlogPost(post);
    });
    const testimonialsData = [
      {
        quote: "After suffering from chronic back pain for years, the team at PhysioForU developed a treatment plan that finally gave me relief. Their expertise and personalized care made all the difference.",
        name: "Sarah Johnson",
        title: "Back Pain Patient",
        rating: 5
      },
      {
        quote: "As a competitive runner, I needed specialized care for my knee injury. The physiotherapists at PhysioForU not only helped me recover but also improved my running technique to prevent future injuries.",
        name: "Michael Davis",
        title: "Sports Injury Patient",
        rating: 5
      },
      {
        quote: "Following my shoulder surgery, I was worried about regaining full movement. The rehabilitation program designed by PhysioForU was comprehensive and effective. I'm now back to all my normal activities.",
        name: "Jennifer Taylor",
        title: "Post-Surgery Patient",
        rating: 4.5
      }
    ];
    testimonialsData.forEach((testimonial) => {
      this.createTestimonial(testimonial);
    });
    const youtubeVideosData = [
      // Only including videos that actually exist on YouTube
      {
        title: "Indian Physician Neurosciences 17th rank Physiotherapist Dr Vishal Sharma (PhysioForU) Jaipur",
        videoId: "owGRfzhxaz0",
        description: "Learn more about Dr. Vishal Sharma's qualifications and expertise in physiotherapy at PhysioForU clinic in Jaipur.",
        thumbnailUrl: "https://i.ytimg.com/vi/owGRfzhxaz0/maxresdefault.jpg",
        category: "About Us"
      },
      {
        title: "Why Physiotherapy is Important",
        videoId: "1l3ugHKC2Aw",
        description: "Discover the importance of physiotherapy in recovery and maintaining physical health.",
        thumbnailUrl: "https://i.ytimg.com/vi/1l3ugHKC2Aw/maxresdefault.jpg",
        category: "Educational"
      },
      {
        title: "Physiotherapy Techniques",
        videoId: "d3j4MGFFqW0",
        description: "Learn about various physiotherapy techniques used in professional treatment.",
        thumbnailUrl: "https://i.ytimg.com/vi/d3j4MGFFqW0/maxresdefault.jpg",
        category: "Techniques"
      },
      {
        title: "Home Exercise Routine for Better Mobility",
        videoId: "ApGhiByLWi8",
        description: "Follow these exercises at home to improve your mobility and flexibility.",
        thumbnailUrl: "https://i.ytimg.com/vi/ApGhiByLWi8/maxresdefault.jpg",
        category: "Home Exercises"
      },
      {
        title: "Preventing Sports Injuries Through Physiotherapy",
        videoId: "xNYUHdPnojs",
        description: "Learn how physiotherapy can help prevent common sports injuries.",
        thumbnailUrl: "https://i.ytimg.com/vi/xNYUHdPnojs/maxresdefault.jpg",
        category: "Sports Injuries"
      },
      {
        title: "Physiotherapy for Chronic Pain Management",
        videoId: "etfLJYM1Lns",
        description: "Discover how physiotherapy techniques can help manage chronic pain conditions.",
        thumbnailUrl: "https://i.ytimg.com/vi/etfLJYM1Lns/maxresdefault.jpg",
        category: "Pain Management"
      },
      {
        title: "Recovering from Joint Replacement Surgery",
        videoId: "etKRfNyK66Y",
        description: "Physiotherapy guidance for recovery after joint replacement surgery.",
        thumbnailUrl: "https://i.ytimg.com/vi/etKRfNyK66Y/maxresdefault.jpg",
        category: "Rehabilitation"
      },
      {
        title: "Stretching Exercises for Neck and Shoulder Pain",
        videoId: "ELZmn_vCOiE",
        description: "Follow these stretches to relieve tension and pain in the neck and shoulder area.",
        thumbnailUrl: "https://i.ytimg.com/vi/ELZmn_vCOiE/maxresdefault.jpg",
        category: "Neck Pain"
      },
      {
        title: "Back Pain Relief Through Gentle Exercise",
        videoId: "YLINHcT_pqQ",
        description: "Learn gentle exercises that can help alleviate back pain without straining your body.",
        thumbnailUrl: "https://i.ytimg.com/vi/YLINHcT_pqQ/maxresdefault.jpg",
        category: "Back Pain"
      },
      {
        title: "Understanding Physiotherapy Assessment",
        videoId: "rL8MnAjykFk",
        description: "What to expect during your first physiotherapy assessment and how it helps in your treatment.",
        thumbnailUrl: "https://i.ytimg.com/vi/rL8MnAjykFk/maxresdefault.jpg",
        category: "Educational"
      },
      // Adding new YouTube videos
      {
        title: "Physiotherapy Jaipur Dr Vishal Sharma Physioforu",
        videoId: "jK2mAW2rjPI",
        description: "Physiotherapy treatment by Dr. Vishal Sharma at PhysioForU clinic in Jaipur.",
        thumbnailUrl: "https://i.ytimg.com/vi/jK2mAW2rjPI/maxresdefault.jpg",
        category: "About Us"
      },
      {
        title: "How to Increase Height with Physiotherapy Exercises",
        videoId: "ibvMpENgX-k",
        description: "Exercises that can potentially help with height increase through improved posture and alignment.",
        thumbnailUrl: "https://i.ytimg.com/vi/ibvMpENgX-k/maxresdefault.jpg",
        category: "Growth & Development"
      },
      {
        title: "Physiotherapy Treatment for Frozen Shoulder",
        videoId: "ZADWd0-iQRw",
        description: "Learn about effective physiotherapy techniques for treating frozen shoulder condition.",
        thumbnailUrl: "https://i.ytimg.com/vi/ZADWd0-iQRw/maxresdefault.jpg",
        category: "Shoulder Pain"
      },
      {
        title: "Physiotherapy Exercises for Knee Pain Relief",
        videoId: "ewq4leNzNQY",
        description: "Guided exercises to help relieve knee pain and improve mobility.",
        thumbnailUrl: "https://i.ytimg.com/vi/ewq4leNzNQY/maxresdefault.jpg",
        category: "Knee Pain"
      },
      {
        title: "Ankle Sprain Physiotherapy Treatment",
        videoId: "Q0xOTHrQzCU",
        description: "Effective physiotherapy treatment methods for ankle sprains and injuries.",
        thumbnailUrl: "https://i.ytimg.com/vi/Q0xOTHrQzCU/maxresdefault.jpg",
        category: "Ankle Injuries"
      },
      {
        title: "Back Pain Relief Exercises - Part 1",
        videoId: "gdFOa32ZoG0",
        description: "First part of a series showing exercises for relieving back pain through proper movement.",
        thumbnailUrl: "https://i.ytimg.com/vi/gdFOa32ZoG0/maxresdefault.jpg",
        category: "Back Pain"
      },
      {
        title: "Home Physiotherapy Techniques for Neck Pain",
        videoId: "vUgV2FCqhRU",
        description: "Simple physiotherapy exercises you can do at home to relieve neck pain.",
        thumbnailUrl: "https://i.ytimg.com/vi/vUgV2FCqhRU/maxresdefault.jpg",
        category: "Neck Pain"
      },
      {
        title: "Treating Tennis Elbow with Physiotherapy",
        videoId: "pPRJhKy91X0",
        description: "Effective physiotherapy techniques for treating tennis elbow and related conditions.",
        thumbnailUrl: "https://i.ytimg.com/vi/pPRJhKy91X0/maxresdefault.jpg",
        category: "Elbow Pain"
      },
      {
        title: "Physiotherapy for Plantar Fasciitis",
        videoId: "ZfvlnPP1Aik",
        description: "Treatment and exercise options for managing plantar fasciitis through physiotherapy.",
        thumbnailUrl: "https://i.ytimg.com/vi/ZfvlnPP1Aik/maxresdefault.jpg",
        category: "Foot Pain"
      },
      {
        title: "Balance Exercises for Vertigo Patients",
        videoId: "sH3ABbZIyi0",
        description: "Specialized balance training exercises for patients experiencing vertigo.",
        thumbnailUrl: "https://i.ytimg.com/vi/sH3ABbZIyi0/maxresdefault.jpg",
        category: "Vertigo"
      },
      {
        title: "Post-Stroke Rehabilitation Exercises",
        videoId: "mgwOWp67S5g",
        description: "Rehabilitation exercises designed specifically for patients recovering from stroke.",
        thumbnailUrl: "https://i.ytimg.com/vi/mgwOWp67S5g/maxresdefault.jpg",
        category: "Rehabilitation"
      },
      {
        title: "Physiotherapy Techniques for Carpal Tunnel Syndrome",
        videoId: "4c8mW7cMCFc",
        description: "Effective physiotherapy exercises and treatments for carpal tunnel syndrome.",
        thumbnailUrl: "https://i.ytimg.com/vi/4c8mW7cMCFc/maxresdefault.jpg",
        category: "Hand Therapy"
      },
      {
        title: "Physiotherapy for Children with Special Needs",
        videoId: "mQTdi072zIA",
        description: "Specialized physiotherapy approaches for children with special needs and developmental conditions.",
        thumbnailUrl: "https://i.ytimg.com/vi/mQTdi072zIA/maxresdefault.jpg",
        category: "Pediatric Therapy"
      },
      {
        title: "Therapeutic Exercises for Arthritis Management",
        videoId: "siCb0LKCSkA",
        description: "Exercise routines designed to help manage arthritis symptoms and improve joint health.",
        thumbnailUrl: "https://i.ytimg.com/vi/siCb0LKCSkA/maxresdefault.jpg",
        category: "Arthritis"
      },
      {
        title: "Posture Correction Techniques",
        videoId: "k-dtl-e82PE",
        description: "Learn proper techniques for correcting posture problems and preventing related pain.",
        thumbnailUrl: "https://i.ytimg.com/vi/k-dtl-e82PE/maxresdefault.jpg",
        category: "Posture"
      },
      {
        title: "Physiotherapy for Pregnancy-Related Back Pain",
        videoId: "g86QgdJnyjw",
        description: "Safe physiotherapy exercises for pregnant women experiencing back pain.",
        thumbnailUrl: "https://i.ytimg.com/vi/g86QgdJnyjw/maxresdefault.jpg",
        category: "Pregnancy Care"
      },
      {
        title: "Sports Performance Enhancement Through Physiotherapy",
        videoId: "cTrGAPPb6qs",
        description: "How physiotherapy techniques can help improve athletic performance and prevent injuries.",
        thumbnailUrl: "https://i.ytimg.com/vi/cTrGAPPb6qs/maxresdefault.jpg",
        category: "Sports Performance"
      }
    ];
    youtubeVideosData.forEach((video) => {
      this.createYoutubeVideo(video);
    });
  }
};
var storage = new MemStorage();

// shared/schema.ts
import { pgTable, text, serial, integer, boolean, timestamp, date } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
var users = pgTable("users", {
  id: serial("id").primaryKey(),
  username: text("username").notNull().unique(),
  password: text("password").notNull()
});
var insertUserSchema = createInsertSchema(users).pick({
  username: true,
  password: true
});
var youtubeVideos = pgTable("youtube_videos", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  videoId: text("video_id").notNull().unique(),
  description: text("description"),
  thumbnailUrl: text("thumbnail_url"),
  category: text("category"),
  publishedAt: timestamp("published_at").notNull().defaultNow()
});
var services = pgTable("services", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  description: text("description").notNull(),
  icon: text("icon").notNull(),
  slug: text("slug").notNull().unique()
});
var insertServiceSchema = createInsertSchema(services).pick({
  title: true,
  description: true,
  icon: true,
  slug: true
});
var blogPosts = pgTable("blog_posts", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  content: text("content").notNull(),
  excerpt: text("excerpt").notNull(),
  imageUrl: text("image_url").notNull(),
  slug: text("slug").notNull().unique(),
  published: boolean("published").notNull().default(true),
  publishedAt: timestamp("published_at").notNull().defaultNow()
});
var insertBlogPostSchema = createInsertSchema(blogPosts).omit({
  id: true,
  publishedAt: true
});
var testimonials = pgTable("testimonials", {
  id: serial("id").primaryKey(),
  quote: text("quote").notNull(),
  name: text("name").notNull(),
  title: text("title").notNull(),
  rating: integer("rating").notNull()
});
var insertTestimonialSchema = createInsertSchema(testimonials).omit({
  id: true
});
var contactMessages = pgTable("contact_messages", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  email: text("email").notNull(),
  phone: text("phone"),
  subject: text("subject").notNull(),
  message: text("message").notNull(),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var insertContactMessageSchema = createInsertSchema(contactMessages).omit({
  id: true,
  createdAt: true
});
var appointments = pgTable("appointments", {
  id: serial("id").primaryKey(),
  name: text("name").notNull(),
  email: text("email").notNull(),
  phone: text("phone").notNull(),
  serviceId: integer("service_id"),
  preferredDate: date("preferred_date").notNull(),
  preferredTime: text("preferred_time").notNull(),
  message: text("message"),
  createdAt: timestamp("created_at").notNull().defaultNow()
});
var insertAppointmentSchema = createInsertSchema(appointments).omit({
  id: true,
  createdAt: true
});
var insertYoutubeVideoSchema = createInsertSchema(youtubeVideos).omit({
  id: true,
  publishedAt: true
});

// server/routes.ts
import { z } from "zod";

// server/email.ts
import nodemailer from "nodemailer";
async function sendEmail(options) {
  try {
    const isConfigured = process.env.EMAIL_USER && process.env.EMAIL_PASSWORD;
    const config = isConfigured ? {
      service: "gmail",
      auth: {
        user: process.env.EMAIL_USER,
        // Your Gmail address
        pass: process.env.EMAIL_PASSWORD
        // Your Gmail App Password (not regular password)
      }
    } : {
      host: "smtp.gmail.com",
      port: 587,
      secure: false,
      auth: {
        user: "drvishalphysioforu@gmail.com",
        pass: "uyum pxkc amfa nsvv"
      }
    };
    const transporter = nodemailer.createTransport(config);
    const from = isConfigured ? process.env.EMAIL_USER : "drvishalphysioforu@gmail.com";
    const info = await transporter.sendMail({
      from,
      // Sender address
      to: options.to,
      // List of receivers
      subject: options.subject,
      // Subject line
      text: options.text,
      // Plain text body
      html: options.html
      // HTML body
    });
    console.log("Email sent successfully:", info.messageId);
    return true;
  } catch (error) {
    console.error("Error sending email:", error);
    return false;
  }
}

// server/routes.ts
async function registerRoutes(app2) {
  const apiRouter = app2.route("/api");
  app2.get("/api/services", async (req, res) => {
    try {
      const services2 = await storage.getServices();
      res.json(services2);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch services" });
    }
  });
  app2.get("/api/services/:slug", async (req, res) => {
    try {
      const service = await storage.getServiceBySlug(req.params.slug);
      if (!service) {
        return res.status(404).json({ message: "Service not found" });
      }
      res.json(service);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch service" });
    }
  });
  app2.get("/api/blog-posts", async (req, res) => {
    try {
      const blogPosts2 = await storage.getBlogPosts();
      res.json(blogPosts2);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch blog posts" });
    }
  });
  app2.get("/api/blog-posts/:slug", async (req, res) => {
    try {
      const post = await storage.getBlogPostBySlug(req.params.slug);
      if (!post) {
        return res.status(404).json({ message: "Blog post not found" });
      }
      res.json(post);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch blog post" });
    }
  });
  app2.get("/api/testimonials", async (req, res) => {
    try {
      const testimonials2 = await storage.getTestimonials();
      res.json(testimonials2);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch testimonials" });
    }
  });
  app2.post("/api/contact", async (req, res) => {
    try {
      const validatedData = insertContactMessageSchema.parse(req.body);
      const message = await storage.createContactMessage(validatedData);
      const userEmailSubject = `Thank you for contacting PhysioForU - We've received your message`;
      const userEmailText = `
Dear ${message.name},

Thank you for reaching out to PhysioForU! We have received your message and appreciate you taking the time to contact us.

Your Message Details:
Subject: ${message.subject}
Message: ${message.message}

We will review your inquiry and get back to you within 24 hours. If this is an urgent matter, please feel free to call us directly at +91 9782219444.

Best regards,
PhysioForU Team
C-98, Om Path, Bhagirath Marg
Shyam Nagar, Behind Community Centre
Jaipur, Rajasthan, India
Phone: +91 9782219444
Email: physioforu5@gmail.com
`;
      const userEmailHtml = `
<!DOCTYPE html>
<html>
<head>
    <style>
        body { font-family: Arial, sans-serif; line-height: 1.6; color: #333; }
        .header { background-color: #2563eb; color: white; padding: 20px; text-align: center; }
        .content { padding: 20px; }
        .message-box { background-color: #f8f9fa; border-left: 4px solid #2563eb; padding: 15px; margin: 20px 0; }
        .footer { background-color: #f8f9fa; padding: 15px; text-align: center; font-size: 14px; color: #666; }
    </style>
</head>
<body>
    <div class="header">
        <h1>PhysioForU</h1>
        <p>Thank you for contacting us!</p>
    </div>
    
    <div class="content">
        <p>Dear ${message.name},</p>
        
        <p>Thank you for reaching out to PhysioForU! We have received your message and appreciate you taking the time to contact us.</p>
        
        <div class="message-box">
            <h3>Your Message Details:</h3>
            <p><strong>Subject:</strong> ${message.subject}</p>
            <p><strong>Message:</strong> ${message.message}</p>
        </div>
        
        <p>We will review your inquiry and get back to you within 24 hours. If this is an urgent matter, please feel free to call us directly at <strong>+91 9782219444</strong>.</p>
        
        <p>Best regards,<br>
        <strong>PhysioForU Team</strong></p>
    </div>
    
    <div class="footer">
        <p>
            C-98, Om Path, Bhagirath Marg<br>
            Shyam Nagar, Behind Community Centre<br>
            Jaipur, Rajasthan, India<br>
            Phone: +91 9782219444 | Email: physioforu5@gmail.com
        </p>
    </div>
</body>
</html>
`;
      const emailSent = await sendEmail({
        to: message.email,
        subject: userEmailSubject,
        text: userEmailText,
        html: userEmailHtml
      });
      res.status(201).json({
        message: "Thank you for your message! We will get back to you soon. A confirmation email has been sent to your email address.",
        id: message.id,
        emailSent
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({
          message: "Invalid form data",
          errors: error.errors
        });
      }
      res.status(500).json({ message: "Failed to submit contact form" });
    }
  });
  app2.get("/api/youtube-videos", async (req, res) => {
    try {
      const videos = await storage.getYoutubeVideos();
      res.json(videos);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch YouTube videos" });
    }
  });
  app2.get("/api/youtube-videos/category/:category", async (req, res) => {
    try {
      const videos = await storage.getYoutubeVideosByCategory(req.params.category);
      res.json(videos);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch YouTube videos by category" });
    }
  });
  app2.get("/api/youtube-videos/:id", async (req, res) => {
    try {
      const id = parseInt(req.params.id, 10);
      if (isNaN(id)) {
        return res.status(400).json({ message: "Invalid video ID" });
      }
      const video = await storage.getYoutubeVideoById(id);
      if (!video) {
        return res.status(404).json({ message: "YouTube video not found" });
      }
      res.json(video);
    } catch (error) {
      res.status(500).json({ message: "Failed to fetch YouTube video" });
    }
  });
  app2.post("/api/youtube-videos", async (req, res) => {
    try {
      const validatedData = insertYoutubeVideoSchema.parse(req.body);
      const video = await storage.createYoutubeVideo(validatedData);
      res.status(201).json({
        message: "YouTube video added successfully",
        id: video.id
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({
          message: "Invalid video data",
          errors: error.errors
        });
      }
      res.status(500).json({ message: "Failed to add YouTube video" });
    }
  });
  app2.post("/api/appointments", async (req, res) => {
    try {
      const validatedData = insertAppointmentSchema.parse(req.body);
      const appointment = await storage.createAppointment(validatedData);
      let serviceName = "General consultation";
      if (appointment.serviceId) {
        const service = await storage.getServiceById(appointment.serviceId);
        if (service) {
          serviceName = service.title;
        }
      }
      const emailSubject = `New Appointment Request: ${appointment.name}`;
      const emailText = `
New appointment request details:

Name: ${appointment.name}
Email: ${appointment.email}
Phone: ${appointment.phone}
Service: ${serviceName}
Preferred Date: ${appointment.preferredDate}
Preferred Time: ${appointment.preferredTime}
${appointment.message ? `Additional Message: ${appointment.message}` : ""}
`;
      const emailHtml = `
<h2>New Appointment Request</h2>
<p><strong>Name:</strong> ${appointment.name}</p>
<p><strong>Email:</strong> ${appointment.email}</p>
<p><strong>Phone:</strong> ${appointment.phone}</p>
<p><strong>Service:</strong> ${serviceName}</p>
<p><strong>Preferred Date:</strong> ${appointment.preferredDate}</p>
<p><strong>Preferred Time:</strong> ${appointment.preferredTime}</p>
${appointment.message ? `<p><strong>Additional Message:</strong> ${appointment.message}</p>` : ""}
`;
      const emailSent = await sendEmail({
        to: "physioforu5@gmail.com",
        subject: emailSubject,
        text: emailText,
        html: emailHtml
      });
      res.status(201).json({
        message: "Your appointment request has been submitted successfully! We will contact you soon to confirm.",
        id: appointment.id,
        emailSent
      });
    } catch (error) {
      if (error instanceof z.ZodError) {
        return res.status(400).json({
          message: "Invalid appointment data",
          errors: error.errors
        });
      }
      console.error("Appointment error:", error);
      res.status(500).json({ message: "Failed to submit appointment request" });
    }
  });
  const httpServer = createServer(app2);
  return httpServer;
}

// server/vite.ts
import express from "express";
import fs from "fs";
import path2, { dirname as dirname2 } from "path";
import { fileURLToPath as fileURLToPath2 } from "url";
import { createServer as createViteServer, createLogger } from "vite";

// vite.config.ts
import { defineConfig } from "vite";
import react from "@vitejs/plugin-react";
import themePlugin from "@replit/vite-plugin-shadcn-theme-json";
import runtimeErrorOverlay from "@replit/vite-plugin-runtime-error-modal";
import path, { dirname } from "path";
import { fileURLToPath } from "url";
var __filename = fileURLToPath(import.meta.url);
var __dirname = dirname(__filename);
var vite_config_default = defineConfig(async () => {
  const plugins = [
    react(),
    runtimeErrorOverlay(),
    themePlugin()
  ];
  if (process.env.NODE_ENV !== "production" && process.env.REPL_ID !== void 0) {
    const { cartographer } = await import("@replit/vite-plugin-cartographer");
    plugins.push(cartographer());
  }
  return {
    base: "/",
    // If deploying to root domain (https://drvishalsharma.com)
    plugins,
    resolve: {
      alias: {
        "@": path.resolve(__dirname, "client", "src"),
        "@shared": path.resolve(__dirname, "shared"),
        "@assets": path.resolve(__dirname, "attached_assets")
      }
    },
    root: path.resolve(__dirname, "client"),
    build: {
      outDir: path.resolve(__dirname, "dist/public"),
      emptyOutDir: true
    }
  };
});

// server/vite.ts
import { nanoid } from "nanoid";
var __filename2 = fileURLToPath2(import.meta.url);
var __dirname2 = dirname2(__filename2);
var viteLogger = createLogger();
function log(message, source = "express") {
  const formattedTime = (/* @__PURE__ */ new Date()).toLocaleTimeString("en-US", {
    hour: "numeric",
    minute: "2-digit",
    second: "2-digit",
    hour12: true
  });
  console.log(`${formattedTime} [${source}] ${message}`);
}
async function setupVite(app2, server) {
  const serverOptions = {
    middlewareMode: true,
    hmr: { server },
    allowedHosts: true
  };
  const vite = await createViteServer({
    ...vite_config_default,
    configFile: false,
    customLogger: {
      ...viteLogger,
      error: (msg, options) => {
        viteLogger.error(msg, options);
        process.exit(1);
      }
    },
    server: serverOptions,
    appType: "custom"
  });
  app2.use(vite.middlewares);
  app2.use("*", async (req, res, next) => {
    const url = req.originalUrl;
    try {
      const clientTemplate = path2.resolve(
        __dirname2,
        "..",
        "client",
        "index.html"
      );
      let template = await fs.promises.readFile(clientTemplate, "utf-8");
      template = template.replace(
        `src="/src/main.tsx"`,
        `src="/src/main.tsx?v=${nanoid()}"`
      );
      const page = await vite.transformIndexHtml(url, template);
      res.status(200).set({ "Content-Type": "text/html" }).end(page);
    } catch (e) {
      vite.ssrFixStacktrace(e);
      next(e);
    }
  });
}
function serveStatic(app2) {
  const distPath = path2.resolve(__dirname2, "public");
  if (!fs.existsSync(distPath)) {
    throw new Error(
      `Could not find the build directory: ${distPath}, make sure to build the client first`
    );
  }
  app2.use(express.static(distPath));
  app2.use("*", (_req, res) => {
    res.sendFile(path2.resolve(distPath, "index.html"));
  });
}

// server/index.ts
var app = express2();
app.use(express2.json());
app.use(express2.urlencoded({ extended: false }));
app.use((req, res, next) => {
  const start = Date.now();
  const path3 = req.path;
  let capturedJsonResponse = void 0;
  const originalResJson = res.json;
  res.json = function(bodyJson, ...args) {
    capturedJsonResponse = bodyJson;
    return originalResJson.apply(res, [bodyJson, ...args]);
  };
  res.on("finish", () => {
    const duration = Date.now() - start;
    if (path3.startsWith("/api")) {
      let logLine = `${req.method} ${path3} ${res.statusCode} in ${duration}ms`;
      if (capturedJsonResponse) {
        logLine += ` :: ${JSON.stringify(capturedJsonResponse)}`;
      }
      if (logLine.length > 80) {
        logLine = logLine.slice(0, 79) + "\u2026";
      }
      log(logLine);
    }
  });
  next();
});
(async () => {
  const server = await registerRoutes(app);
  app.use((err, _req, res, _next) => {
    const status = err.status || err.statusCode || 500;
    const message = err.message || "Internal Server Error";
    res.status(status).json({ message });
    throw err;
  });
  if (app.get("env") === "development") {
    await setupVite(app, server);
  } else {
    serveStatic(app);
  }
  const port = process.env.PORT || 5e3;
  const host = process.env.NODE_ENV === "production" ? "0.0.0.0" : "localhost";
  server.listen({
    port,
    host
  }, () => {
    log(`serving on ${host}:${port}`);
  });
})();
